package hexdump;

import java.io.*;
import java.util.*;

/**
 * The {@code HexDump} class provides a client for displaying the contents of a
 * binary file in hexadecimal.
 * <p>
 *
 * @author Eddie Vanda
 */
public class HexDump {

  private static String usage = "Usage: \n HexDump inputFile\n";
  private static String prompt = "\nHexDump Menu:"
          + "\n 0-9999 to select file or folder after d:"
          + "\n a: enter next hexdump hexidecimal address"
          + "\n d: show directories and files in current folder"
          + "\n h: hex dump next 16 bytes - optionally decimal line count follows"
          + "\n o: octal dump next 16 bytes - optionally decimal line count follows"
          + "\n s: enter silent mode (no prompts)"
          + "\n ?: exit silent mode"
          + "\n u: go up to parent folder"
          + "\n x: exit hexdump program"
          + "\n  press number or letter for required function: ";

  private FileIn fi;
  private String fileName;
  private int dumpAddress;
  private ArrayList<File> dirList;
  boolean silent = false;
  private int format = 0; // 0 = hex, 1 = octal, 2 = binary

  private HexDump(String fileName) {
    this.fileName = fileName;
    fi = new FileIn(this, fileName);
    if (fi.isOk()) {
      run();
    }
  }

  private void run() {
    Scanner kb = new Scanner(System.in);
    boolean done = false;
    boolean isNumeric = false;
    while (!done) {
      if (!silent) {
        //System.out.println("Current Folder: " + FileIn.getCurrentFolder());
        System.out.println("Current File:     " + FileIn.getCurrentFile());
        System.out.print(prompt);
      }
      String s = kb.nextLine();
      if (s.length() > 0) {
        try {
          int dirChoice = Integer.parseInt(s);
          select(dirChoice);
          isNumeric = true;
        } catch (Exception e) {
          isNumeric = false;
        }
        if (!isNumeric) {
          done = doChoice(s);
        }
      }
    }

  }

  private boolean doChoice(String s) {
    boolean done = false;
    switch (s.charAt(0)) {
      case 'a':
        addressChange(s);
        break;
      case 'b':
        doDump(s, 2);
        break;
      case 'h':
        doDump(s, 0);
        break;
      case 'o':
        doDump(s, 1);
        break;
      case 'd':
        showDir();
        break;
      case 's':
        silent = true;
        break;
      case 'x':
        done = true;
        break;
      case 'u':
        goUp();
        break;
      case '?':
        silent = false;
        break;
      default:
        System.out.println("Could not understand " + s);
    }
    return done;
  }

  private void goUp() {
    fi.setCurrentFolder(FileIn.getParent(fi.getCurrentFolder()));
    FileIn.setCurrentFile(null);
    showDir();
  }

  private void select(int fileItem) {
    try {
      File file = dirList.get(fileItem);
      System.out.println("file: " + file);
      if (file.isDirectory()) {
        fi.setCurrentFolder(file);
        FileIn.setCurrentFile(null);
        showDir();
      } else {
        dumpAddress = 0;
        fi.setCurrentFile(file);
      }
    } catch (Exception e) {
      System.out.println(e.toString());
    }
  }
  
  
  
  
  private void showDir() {
    dirList = FileIn.loadFoldersAndFiles();
    System.out.println("\nDir llisting of " + FileIn.getCurrentFolder().toString() + "\n");
    int count = 0;
    for (File file : dirList) {
      System.out.printf(" %4d ", (count++));
      if (file.isDirectory()) {
        System.out.print("d ");
      } else {
        if(file.isFile()) {
        System.out.print("f ");
        System.out.printf(" %7d ", file.length());
        System.out.print(" ");
        } else
          System.out.println("Neither d nor f : ");
      }
      System.out.println(FileIn.getFileString(file));
    }
    System.out.println();
  }

 
  private void addressChange(String s) {
    String s1 = s.substring(1);
     dumpAddress = hex2Int(s1);
  }

  private void doDump(String s, int format) {
    if (s.length() == 1) {
      dumpOneLine(format);
    } else {
      try {
        s = s.substring(1);
        s = s.strip();
        System.out.println("doDump line repeat: <" + s + ">");
        int count = Integer.parseInt(s);
        while ((count--) > 0) {
          dumpOneLine(format);
        }
      } catch (Exception e) {
        System.out.println(e.toString());
      }
    }
  }

  private void dumpOneLine(int format) {
    byte[] bytes = fi.getByteRow(dumpAddress, format);
    if (bytes != null) {
      System.out.printf("%06x:", dumpAddress);
      dumpAddress += bytes.length;
      for (int i = 0; i < bytes.length; i++) {
        try {
          if ((i % 4) == 0) {
            System.out.print(" ");
          }
          switch (format) {
            case 0: 
              System.out.printf("%02x ", bytes[i]);
              break;
            case 1: 
              System.out.printf("%03o ", bytes[i]);
              break;
            case 2: 
              System.out.print(getBinary (bytes[i]));
              break;
          }
        } catch (Exception e) {
          System.out.println(e.toString());
        }
      }
      System.out.print(" - ");
      for (byte b : bytes) {
        if (b > 0x1F && b < 0x7F) {
          System.out.print((char) b);
        } else {
          System.out.print(".");
        }
      }
      System.out.println();
    }
  }

  /**
   * test input mechanism
   */
  public static void main(String[] args) {
    if (args.length == 1) {
      new HexDump(args[0]);
    } else {
      System.out.println(usage);
    }

  }
  
  
  private String getBinary(byte b){
    StringBuilder answer = new StringBuilder();
    for (int i = 0; i < 8; i++) {
      if (b < 0)
        answer.append("1");
      else
        answer.append("0");
      b = (byte)(b * 2);
    }
    return answer.toString() + " ";
  }
  /**
   * hex2Int
   * @param s hex string
   * @return int value
   */
  private int hex2Int (String s){
    String ref = "0123456789abcdef";
    int answer = 0;
    s = s.strip().toLowerCase();
    if(s.length() > 0){
      for (int i = 0; i < s.length(); i++){
        int ival = ref.indexOf(s.charAt(i));
        if (ival > -1){
          answer = answer * 16 + ival;
        }
      }
    }
    return answer;
  }
}
